#include <stdio.h>

#include "AudioOutput.h"
#include "playcas.h"
#include "sofacas.h"

static int	g_iDeviceIndex;

/*
 =======================================================================================================================
 =======================================================================================================================
 */
static void vChecksSound(HRESULT _hResult)
{
	if(_hResult != DS_OK) vMyError(-1, "Audio error %d.", _hResult);
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
BOOL CALLBACK AudioOutput::bEnumerateCallback(LPGUID lpGUID, LPCTSTR lpszDesc, LPCTSTR lpszDrvName, LPVOID lpContext)
{
	/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
	AudioOutput *thiz = (AudioOutput *) lpContext;
	char		*szFixedDesc;
	/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

	szFixedDesc = szFixedString(lpszDesc);

	if(thiz)
	{
		if(g_iDeviceIndex++ == thiz->m_iDesiredDeviceIndex)
		{
			printf("Output device   : %s\n", szFixedDesc);
			printf("\nPress [ESC] to stop playing\n\n");
			thiz->m_poDesiredDeviceGUID = lpGUID;
			thiz->m_bDeviceFound = true;
		}
	}
	else
	{
		printf("    %d: %s\n", g_iDeviceIndex++, szFixedDesc);
	}

	free(szFixedDesc);
	return(TRUE);
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void AudioOutput::vListAudioDevices()
{
	g_iDeviceIndex = 0;
	printf("Audio devices:\n");
	DirectSoundEnumerate((LPDSENUMCALLBACK) bEnumerateCallback, NULL);
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
AudioOutput::AudioOutput
(
	int		_iFrequency,
	int		_iBytesPerSample,
	double	_fBufferLength,
	char	*_szOutputFileName,
	int		_iOutputDevice
)
{
	g_iDeviceIndex = 0;
	m_iDesiredDeviceIndex = _iOutputDevice;
	m_poDesiredDeviceGUID = NULL;
	m_poStreamBuffer = NULL;
	m_poPrimaryBuffer = NULL;
	m_poOutputFile = NULL;
	m_iFrequency = _iFrequency;
	m_iBytesPerSample = _iBytesPerSample;
	m_uiWritten = 0;
	m_iHWBytesAvailable = 0;
	m_iHWPreviousBytesAvailable = 0;

	if(_szOutputFileName)
	{
		m_poOutputFile = myfopen(_szOutputFileName, true);
		g_oWaveHeader.m_Frequence = _iFrequency;
		g_oWaveHeader.m_BytePerSec = _iFrequency * m_iBytesPerSample;
		g_oWaveHeader.m_BytePerBloc = m_iBytesPerSample;
		g_oWaveHeader.m_BitsPerSample = m_iBytesPerSample * 8;
		fwrite(&g_oWaveHeader, 1, sizeof(g_oWaveHeader), m_poOutputFile);
	}
	else
	{
		/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
		DWORD			iLockedSize;
		DSBUFFERDESC	oPrimaryDesc;
		DSBUFFERDESC	oStreamDesc;
		WAVEFORMATEX	oObtainedFormat;
		WAVEFORMATEX	oRequestedFormat;
		/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

		m_bDeviceFound = false;
		DirectSoundEnumerate((LPDSENUMCALLBACK) bEnumerateCallback, this);

		if(!m_bDeviceFound)
		{
			vMyError(-1, "Unable to open audio device #%d.\n", m_iDesiredDeviceIndex);
		}

		m_iHWBufferSize = (int) (_fBufferLength * m_iFrequency) * m_iBytesPerSample;

		vChecksSound(DirectSoundCreate(m_poDesiredDeviceGUID, &m_poDirectSound, NULL));
		vChecksSound(m_poDirectSound->SetCooperativeLevel(GetConsoleWindow(), DSSCL_EXCLUSIVE));

		/* Make a format description for what we want */
		oRequestedFormat.wBitsPerSample = m_iBytesPerSample * 8;
		oRequestedFormat.wFormatTag = WAVE_FORMAT_PCM;
		oRequestedFormat.nChannels = 1;
		oRequestedFormat.nSamplesPerSec = _iFrequency;
		oRequestedFormat.nBlockAlign = m_iBytesPerSample * oRequestedFormat.nChannels;
		oRequestedFormat.nAvgBytesPerSec = oRequestedFormat.nSamplesPerSec * oRequestedFormat.nBlockAlign;
		oRequestedFormat.cbSize = 0;

		/* Create a buffer desc for the primary buffer */
		memset(&oPrimaryDesc, 0, sizeof(oPrimaryDesc));
		oPrimaryDesc.dwSize = sizeof(oPrimaryDesc);
		oPrimaryDesc.dwFlags = DSBCAPS_PRIMARYBUFFER | DSBCAPS_GETCURRENTPOSITION2;
		oPrimaryDesc.lpwfxFormat = NULL;

		vChecksSound(m_poDirectSound->CreateSoundBuffer(&oPrimaryDesc, &m_poPrimaryBuffer, NULL));
		vChecksSound(m_poPrimaryBuffer->SetFormat(&oRequestedFormat));

		/* Create a buffer desc for the stream buffer */
		memset(&oStreamDesc, 0, sizeof(oStreamDesc));
		oStreamDesc.dwSize = sizeof(oStreamDesc);
		oStreamDesc.dwFlags = DSBCAPS_GLOBALFOCUS | DSBCAPS_GETCURRENTPOSITION2;
		oStreamDesc.dwBufferBytes = m_iHWBufferSize;
		oStreamDesc.lpwfxFormat = &oRequestedFormat;

		/* Create the stream buffer */
		vChecksSound(m_poDirectSound->CreateSoundBuffer(&oStreamDesc, &m_poStreamBuffer, NULL));

		vChecksSound(m_poStreamBuffer->GetFormat(&oObtainedFormat, sizeof(oObtainedFormat), NULL));
		m_iFrequency = oObtainedFormat.nSamplesPerSec;

		if
		(
			(oObtainedFormat.nChannels != oRequestedFormat.nChannels)
		||	(oObtainedFormat.wBitsPerSample != oRequestedFormat.wBitsPerSample)
		||	(oObtainedFormat.wFormatTag != oRequestedFormat.wFormatTag)
		||	(oObtainedFormat.nBlockAlign != oRequestedFormat.nBlockAlign)
		||	(oObtainedFormat.wBitsPerSample != oRequestedFormat.wBitsPerSample)
		) vMyError(-1, "Unable to create suitable sound buffer.");

		vChecksSound(m_poStreamBuffer->Lock(0, m_iHWBufferSize, (void **) &m_pcHWBuffer, &iLockedSize, NULL, NULL, 0));

		if(iLockedSize != m_iHWBufferSize) vMyError(-1, "Unable to lock entire sound buffer.");
	}
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
AudioOutput::~AudioOutput()
{
	if(m_poOutputFile)
	{
		/*~~~~~~~~~~~~~~~~~~~~~~~*/
		unsigned int	uiFileSize;
		/*~~~~~~~~~~~~~~~~~~~~~~~*/

		uiFileSize = ftell(m_poOutputFile);
		g_oWaveHeader.m_FileSize = uiFileSize - 8;
		g_oWaveHeader.m_DataSize = uiFileSize - sizeof(g_oWaveHeader);
		fseek(m_poOutputFile, 0, SEEK_SET);
		fwrite(&g_oWaveHeader, 1, sizeof(g_oWaveHeader), m_poOutputFile);

		fclose(m_poOutputFile);
	}
	else
	{
		vStop();
		vChecksSound(m_poStreamBuffer->Unlock(m_pcHWBuffer, m_iHWBufferSize, NULL, 0));

		if(m_poStreamBuffer) vChecksSound(m_poStreamBuffer->Release());
		if(m_poPrimaryBuffer) vChecksSound(m_poPrimaryBuffer->Release());
		if(m_poDirectSound) vChecksSound(m_poDirectSound->Release());
	}
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
int AudioOutput::iFrequency()
{
	return m_iFrequency;
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void AudioOutput::vPlay()
{
	/*~~~~~~~~~~~~~~~~~~*/
	DWORD	iPlayPosition;
	/*~~~~~~~~~~~~~~~~~~*/

	if(m_poStreamBuffer)
	{
		vChecksSound(m_poStreamBuffer->Play(0, 0, DSBPLAY_LOOPING));
		Sleep(1);

		vChecksSound(m_poStreamBuffer->GetCurrentPosition((LPDWORD) & iPlayPosition, NULL));
		m_iHWPosition = iPlayPosition;
		m_iHWBytesAvailable = 0;
		m_iHWPreviousBytesAvailable = 0;
	}
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void AudioOutput::vRender(double _fSample)
{
	if(_fSample < -1.0)
		_fSample = -1.0;
	else if(_fSample > 1.0)
		_fSample = 1.0;

	if(m_poOutputFile)
	{
		switch(m_iBytesPerSample)
		{
		case 1:
			{
				/*~~~~~~~~~~~~~~~~~~~~~*/
				unsigned char	ucSample;
				/*~~~~~~~~~~~~~~~~~~~~~*/

				ucSample = (unsigned char) (((_fSample + 1.0) * 0.5) * 255.0);
				fwrite(&ucSample, 1, sizeof(ucSample), m_poOutputFile);
			}
			break;

		case 2:
			{
				/*~~~~~~~~~~~~~~~~*/
				short int	iSample;
				/*~~~~~~~~~~~~~~~~*/

				iSample = (short int) (_fSample * 32767.0);
				fwrite(&iSample, 1, sizeof(iSample), m_poOutputFile);
			}
			break;
		}

		m_uiWritten++;
	}
	else
	{
		if(m_poStreamBuffer)
		{
			do
			{
				while(!m_iHWBytesAvailable && bNoUserStop())
				{
					/*~~~~~~~~~~~~~~~~~*/
					DWORD	iMaxPosition;
					/*~~~~~~~~~~~~~~~~~*/

					vChecksSound(m_poStreamBuffer->GetCurrentPosition(&iMaxPosition, NULL));

					if((signed) iMaxPosition < m_iHWPosition)
					{
						iMaxPosition += m_iHWBufferSize;
					}

					m_iHWBytesAvailable = iMaxPosition - m_iHWPosition;

					if(!m_iHWBytesAvailable)
					{
						vUpdateUserStop();
						Sleep(1);
					}
					else
					{
						if(m_iHWPreviousBytesAvailable > m_iHWBytesAvailable + m_iHWBufferSize / 2)
						{
							vMyError(-1, "Hardware buffer underrrun.");
						}

						m_iHWPreviousBytesAvailable = m_iHWBytesAvailable;
					}
				}

				switch(m_iBytesPerSample)
				{
				case 1:
					{
						/*~~~~~~~~~~~~~~~~~~~~~*/
						unsigned char	ucSample;
						/*~~~~~~~~~~~~~~~~~~~~~*/

						ucSample = (unsigned char) (((_fSample + 1.0) * 0.5) * 255.0);
						memcpy(m_pcHWBuffer + m_iHWPosition, &ucSample, sizeof(ucSample));
					}
					break;

				case 2:
					{
						/*~~~~~~~~~~~~~~~~*/
						short int	iSample;
						/*~~~~~~~~~~~~~~~~*/

						iSample = (short int) (_fSample * 32767.0);
						memcpy(m_pcHWBuffer + m_iHWPosition, &iSample, sizeof(iSample));
					}
					break;
				}

				m_iHWBytesAvailable -= m_iBytesPerSample;
				m_iHWPosition += m_iBytesPerSample;

				if(m_iHWPosition >= m_iHWBufferSize)
				{
					m_iHWPosition = 0;
				}
			} while((!g_bREMOn) && bNoUserStop());
		}
	}
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void AudioOutput::vStop()
{
	if(m_poStreamBuffer)
	{
		/*~~~~~~~*/
		int iIndex;
		/*~~~~~~~*/

		for(iIndex = 0; iIndex < m_iHWBufferSize / (signed) m_iBytesPerSample; iIndex++)
		{
			vRender(0);
		}

		vChecksSound(m_poStreamBuffer->Stop());
	}
}
