#include <stdio.h>

#include "AudioInput.h"
#include "reccas.h"
#include "sofacas.h"

static int	g_iDeviceIndex;

/*
 =======================================================================================================================
 =======================================================================================================================
 */
static void vChecksSound(HRESULT _hResult)
{
	if(_hResult != DS_OK) vMyError(-1, "Audio error %d.", _hResult);
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
BOOL CALLBACK AudioInput::bEnumerateCallback(LPGUID lpGUID, LPCTSTR lpszDesc, LPCTSTR lpszDrvName, LPVOID lpContext)
{
	/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
	AudioInput	*thiz = (AudioInput *) lpContext;
	char	*szFixedDesc;
	/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

	szFixedDesc = szFixedString(lpszDesc);

	if(thiz)
	{
		if(g_iDeviceIndex++ == thiz->m_iDesiredDeviceIndex)
		{
			printf("Input device : %s\n", szFixedDesc);
			printf("\nPress [ESC] to stop recording\n\n");

			thiz->m_poDesiredDeviceGUID = lpGUID;
			thiz->m_bDeviceFound = true;
		}
	}
	else
	{
		printf("    %d: %s\n", g_iDeviceIndex++, szFixedDesc);
	}

	free(szFixedDesc);
	return TRUE;
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void AudioInput::vListAudioDevices()
{
	g_iDeviceIndex = 0;
	printf("Audio devices:\n");
	DirectSoundCaptureEnumerate((LPDSENUMCALLBACK) bEnumerateCallback, NULL);
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
AudioInput::AudioInput(FILE *_poFile, double _fVolumeAmplification)
{
	m_fMaxAmplitude = 0.0;

	m_poDirectSound = NULL;
	m_poCaptureBuffer = NULL;

	m_piHWBuffer = NULL;
	m_iHWBufferSize = 0;
	m_iHWBytesAvailable = 0;
	m_iHWPreviousBytesAvailable = 0;
	m_iHWPosition = 0;
	m_pcSWBuffer = NULL;
	m_iSWBufferSize = 0;
	m_iSWPosition = 0;
	m_iHead = 0;
	m_fTime = 0.0;

	m_bDeviceFound = false;
	m_iDesiredDeviceIndex = 0;
	m_poDesiredDeviceGUID = NULL;

	m_poWavFile = _poFile;
	m_fVolumeAmplification = _fVolumeAmplification;

	fread(&g_oWaveHeader, sizeof(g_oWaveHeader), 1, m_poWavFile);
	m_iFrequency = g_oWaveHeader.m_Frequence;
	m_iBytesPerBloc = g_oWaveHeader.m_BytePerBloc;
	m_iBitsPerSample = g_oWaveHeader.m_BitsPerSample;

	m_pcBloc = (unsigned char *) malloc(m_iBytesPerBloc);
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
AudioInput::AudioInput
(
	unsigned int	_uiFrequency,
	double			_fVolumeAmplification,
	double			_fBufferLength,
	int				_iInputDevice
)
{
	/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
	DSCBUFFERDESC	oStreamDesc;
	WAVEFORMATEX	oRequestedFormat;
	WAVEFORMATEX	oObtainedFormat;
	DWORD			iLockedSize;
	/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

	m_iDesiredDeviceIndex = _iInputDevice;
	m_iHWPosition = 0;
	m_fMaxAmplitude = 0.0;
	m_fVolumeAmplification = _fVolumeAmplification;
	m_fTime = 0.0;


	/* Make a format description for what we want */
	oRequestedFormat.wBitsPerSample = sizeof(RecAudioSample) * 8;
	oRequestedFormat.wFormatTag = WAVE_FORMAT_PCM;
	oRequestedFormat.nChannels = 1;
	oRequestedFormat.nSamplesPerSec = _uiFrequency;
	oRequestedFormat.nBlockAlign = sizeof(RecAudioSample) * oRequestedFormat.nChannels;
	oRequestedFormat.nAvgBytesPerSec = oRequestedFormat.nSamplesPerSec * oRequestedFormat.nBlockAlign;
	oRequestedFormat.cbSize = 0;

	g_iDeviceIndex = 0;
	m_bDeviceFound = false;
	DirectSoundCaptureEnumerate((LPDSENUMCALLBACK) bEnumerateCallback, this);

	if(!m_bDeviceFound)
	{
		vMyError(-1, "Unable to open audio device #%d.\n", m_iDesiredDeviceIndex);
	}

	vChecksSound(DirectSoundCaptureCreate(m_poDesiredDeviceGUID, &m_poDirectSound, NULL));

	m_iHWBufferSize = (int) (_fBufferLength * _uiFrequency) * oRequestedFormat.nChannels * sizeof(RecAudioSample);

	oStreamDesc.dwSize = sizeof(DSCBUFFERDESC);
	oStreamDesc.dwFlags = 0;
	oStreamDesc.dwBufferBytes = m_iHWBufferSize;
	oStreamDesc.dwReserved = 0;
	oStreamDesc.lpwfxFormat = &oRequestedFormat;
	oStreamDesc.dwFXCount = 0;
	oStreamDesc.lpDSCFXDesc = NULL;

	vChecksSound(m_poDirectSound->CreateCaptureBuffer(&oStreamDesc, &m_poCaptureBuffer, NULL));

	vChecksSound(m_poCaptureBuffer->GetFormat(&oObtainedFormat, sizeof(oObtainedFormat), NULL));
	m_iFrequency = oObtainedFormat.nSamplesPerSec;

	if
	(
		(oObtainedFormat.nChannels != oRequestedFormat.nChannels)
	||	(oObtainedFormat.wBitsPerSample != oRequestedFormat.wBitsPerSample)
	||	(oObtainedFormat.wFormatTag != oRequestedFormat.wFormatTag)
	||	(oObtainedFormat.nBlockAlign != oRequestedFormat.nBlockAlign)
	||	(oObtainedFormat.wBitsPerSample != oRequestedFormat.wBitsPerSample)
	) vMyError(-1, "Unable to create suitable sound buffer.");

	vChecksSound(m_poCaptureBuffer->Lock(0, m_iHWBufferSize, (void **) &m_piHWBuffer, &iLockedSize, NULL, NULL, 0));

	if(iLockedSize != m_iHWBufferSize)
	{
		vMyError(-1, "Unable to lock entire sound buffer.");
	}

	m_iSWBufferSize = 4096;
	m_pcSWBuffer = (char *) malloc(m_iSWBufferSize);
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void AudioInput::vRecord()
{
	if(m_poCaptureBuffer)
	{
		/*~~~~~~~~~~~~~~~~~~*/
		DWORD	iReadPosition;
		/*~~~~~~~~~~~~~~~~~~*/

		m_iHead = -1;

		vChecksSound(m_poCaptureBuffer->Start(DSCBSTART_LOOPING));
		Sleep(1);

		vChecksSound(m_poCaptureBuffer->GetCurrentPosition(NULL, (LPDWORD) & iReadPosition));
		m_iHWPosition = iReadPosition;
		m_iHWBytesAvailable = 0;
		m_iHWPreviousBytesAvailable = 0;

		m_fMaxAmplitude = 0;
		m_iSWPosition = 0;
		m_iHead = 0;
	}
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
bool AudioInput::bRead(double *_pfSample)
{
	/*~~~~~~~~~~~~~~~~~*/
	bool	bEOF = false;
	/*~~~~~~~~~~~~~~~~~*/

	if(m_poCaptureBuffer)
	{
		/*~~~~~~~~*/
		int iResult;
		/*~~~~~~~~*/

		while(!m_iHWBytesAvailable && !bEOF)
		{
			/*~~~~~~~~~~~~~~~~~*/
			DWORD	iMaxPosition;
			/*~~~~~~~~~~~~~~~~~*/

			vChecksSound(m_poCaptureBuffer->GetCurrentPosition(NULL, &iMaxPosition));

			if((signed) iMaxPosition < m_iHWPosition)
			{
				iMaxPosition += m_iHWBufferSize;
			}

			m_iHWBytesAvailable = iMaxPosition - m_iHWPosition;

			if(!m_iHWBytesAvailable)
			{
				vUpdateUserStop();
				if(!bNoUserStop()) bEOF = true;
				Sleep(1);
			}
			else
			{
				if(m_iHWPreviousBytesAvailable > m_iHWBytesAvailable + m_iHWBufferSize / 2)
				{
					vMyError(-1, "Hardware buffer underrrun.");
				}

				m_iHWPreviousBytesAvailable = m_iHWBytesAvailable;
			}
		}

		iResult = m_piHWBuffer[m_iHWPosition / sizeof(RecAudioSample)];

		m_iHWBytesAvailable -= sizeof(RecAudioSample);
		m_iHWPosition += sizeof(RecAudioSample);

		if(m_iHWPosition >= m_iHWBufferSize)
		{
			m_iHWPosition = 0;
		}

		*_pfSample = (double) iResult / 32768.0;
	}
	else
	{
		/*~~~~~~~~~~~~*/
		int iSample = 0;
		/*~~~~~~~~~~~~*/

		vUpdateUserStop();

		if(!bNoUserStop() || (fread(m_pcBloc, m_iBytesPerBloc, 1, m_poWavFile) != 1))
		{
			bEOF = true;
		}
		else
		{
			switch(m_iBitsPerSample)
			{
			case 8:
				memcpy(&iSample, m_pcBloc, 1);
				*_pfSample = iSample - 128;
				*_pfSample /= 128.0;
				break;

			case 16:
				memcpy(&iSample, m_pcBloc, 2);
				if(iSample >= (1 << 15)) iSample = iSample - (1 << 16);
				*_pfSample = iSample;
				*_pfSample /= (double) (1 << 15);
				break;

			case 24:
				memcpy(&iSample, m_pcBloc, 3);
				if(iSample >= (1 << 23)) iSample = iSample - (1 << 24);
				*_pfSample = iSample;
				*_pfSample /= (double) (1 << 23);
				break;
			}
		}
	}

	*_pfSample *= m_fVolumeAmplification;

	if(m_fMaxAmplitude < *_pfSample)
	{
		m_fMaxAmplitude = *_pfSample;
	}
	else if(m_fMaxAmplitude < -*_pfSample)
	{
		m_fMaxAmplitude = -*_pfSample;
	}

	m_fTime += 1.0 / (double)m_iFrequency;

	return !bEOF;
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
AudioInput::~AudioInput()
{
	if(m_poCaptureBuffer)
	{
		vStop();
		vChecksSound(m_poCaptureBuffer->Unlock(m_piHWBuffer, m_iHWBufferSize, NULL, 0));

		vChecksSound(m_poCaptureBuffer->Release());
		vChecksSound(m_poDirectSound->Release());
	}
	else
	{
		fclose(m_poWavFile);
		free(m_pcBloc);
	}
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
int AudioInput::iFrequency()
{
	return m_iFrequency;
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void AudioInput::vStop()
{
	if(m_poCaptureBuffer)
	{
		vChecksSound(m_poCaptureBuffer->Stop());
	}
}
