#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include <utils.h>
#include <ctype.h>

#include "unapi.h"
#include "gr8net.h"
#include "seg.h"
#include "ftpparse.h"

enum
{
	COMMAND_CD		= 0,
	COMMAND_DIR,
	COMMAND_GET,
	COMMAND_GETU,
	COMMAND_GETR,
	COMMAND_GETRU,
	COMMAND_HELP,
	COMMAND_LCD,
	COMMAND_LDIR,
	COMMAND_PUT,
	COMMAND_PUTU,
	COMMAND_PUTR,
	COMMAND_PUTRU,
	COMMAND_QUIT,
	COMMAND_TA,
	COMMAND_VERBOSE,
	COMMAND_LOWER
};

static char *g_acSupportedCommands[] =
{
	"cd", "dir", "get", "getu", "getr", "getru", "help", "lcd", "ldir", "put", "putu", "putr",
	"putru", "quit", "ta", "verbose", "lower"
};

static char *g_szUsage = "\
\r\n\
SofaFTP 1.3 by Louthrax\r\n\
\r\n\
SofaFTP is an MSX-DOS 2 FTP client\r\n\
for GR8NET and UNAPI interfaces\r\n\
\r\n\
FTP server has to support STAT\r\n\
command and passive mode. It must\r\n\
also list directories in UNIX mode\r\n\
\r\n\
Usage: SFTP <server> [commands] [/U]\r\n\
\r\n\
<server> is an FTP server\r\n\
\r\n\
[commands] is an optional list of\r\n\
FTP commands separated by \"!\",\r\n\
with spaces replaced by \"*\"\
\r\n\
[/U] forces use of UNAPI interface\
";

static char *g_szCommandsList = "\
\r\n\
SofaFTP specific commands:\r\n\
\r\n\
CD [remote_dir]\r\n\
  Go to [remote_dir] on\r\n\
  server, or display current\r\n\
  directory if not specified\r\n\
\r\n\
DIR [remote_dir]\r\n\
  List content of [remote_dir] on\r\n\
  server, or current directory if\r\n\
  not specified\r\n\
\r\n\
GET <remote_file> [local_file]\r\n\
  Get <remote_file> from server to\r\n\
  current directory on local\r\n\
  machine, or to [local_file] if\r\n\
  specified\r\n\
\r\n\
GETU <remote_file> [local_file]\r\n\
  Same as GET but only if\r\n\
  [local_file] is older than\r\n\
  <remote_file>\r\n\
\r\n\
GETR [remote_dir] [local_dir]\r\n\
  Recursively get directory from\r\n\
  server to directory on local\r\n\
  machine.It not specified, current\r\n\
  directories are used\r\n\
\r\n\
GETRU [remote_dir] [local_dir]\r\n\
  Same as GETR but only if local\r\n\
  files are older than remote\r\n\
  files\r\n\
\r\n\
HELP\r\n\
  Display this help, plus remote\r\n\
  help\r\n\
\r\n\
LCD [local_dir]\r\n\
  Go to [local_dir] on local\r\n\
  machine, or display current\r\n\
  directory if not specified\r\n\
\r\n\
LDIR [local_dir]\r\n\
  List content of [local_dir] on\r\n\
  local machine, or current\r\n\
  directory if not specified\r\n\
\r\n\
PUT <local_file> [remote_file]\r\n\
  Put <local_file> from local\r\n\
  machine to current directory on\r\n\
  server, or to [remote_file] if\r\n\
  specified\r\n\
\r\n\
PUTU <local_file> [remote_file]\r\n\
  Same as PUT but only if\r\n\
  [remote_file] is older than\r\n\
  <local_file>\r\n\
\r\n\
PUTR [local_dir] [remote_dir]\r\n\
  Recursively put directory from\r\n\
  local machine to directory\r\n\
  on server. It not specified,\r\n\
  current directories are used\r\n\
\r\n\
PUTRU [local_dir] [remote_dir]\r\n\
  Same as PUTR but only if remote\r\n\
  files are older than local files\r\n\
\r\n\
QUIT\r\n\
  Disconnect from server and quit\r\n\
\r\n\
TA <hours>[:<minutes>]\r\n\
  Adjust times obtained from server,\r\n\
  or display current value if not\r\n\
  specified\r\n\
\r\n\
VERBOSE [0|1]\r\n\
  Set verbose mode to ON (1) or OFF\r\n\
  (0), or display current mode if\r\n\
  not specified\r\n\
LOWER [0|1]\r\n\
  Force lower case when creating\r\n\
  remote files or directories\r\n\
";

#define TRANSFER_BLOCK_SIZE			2048
#define MAX_STATUS_LINE_LENGTH	256
#define STATUS_BUFFER_SIZE		2048

typedef struct tdEntry
{
	struct tdEntry	*m_poNext;
	bool			m_bIsDirectory;
	unsigned long	m_ulSize;
	unsigned long	m_ulFileTime;
	char			m_acName[]; /* Keep this one in last position */
} tdEntry;

/* Leave those grouped and uninitialized */
static unsigned char	g_ucCommandCapacity;
static unsigned char	g_ucCommandSize;
static char				g_acCommand[256];

static char				g_acStatusLine[MAX_STATUS_LINE_LENGTH];
static char				*g_acStatusBuffer;
static char				*g_acTransferBuffer;
static char				*g_pcStatus;
static bool				g_bVerbose;
static bool				g_bForceLowerCase;

static char				*g_szHostName = NULL;
static bool				g_bDisplayErrors = true;
static unsigned int		g_iStatusBytesAvailable = 0;
static bool				g_bQuit = false;

static char				g_cHoursAdjust = 0;
static char				g_cMinutesAdjust = 0;

static FILE				*g_poFile = NULL;
static unsigned char	g_ucConnection0 = 0;
static unsigned char	g_ucConnection1 = 0;

static char				*g_acBuffer;
static char				*g_acDataLine;

typedef enum { DEVICE_UNAPI, DEVICE_GR8NET, DEVICE_NONE } tdDeviceType;

tdDeviceType g_eDeviceType = DEVICE_NONE;

tdEntry *poGetDirectoryEntries(char *_szPath);

/*
 =======================================================================================================================
 =======================================================================================================================
 */
unsigned char DRIVER_ucConnectToIPAddress(unsigned int _uiPort, unsigned char *_aucIPAddress)
{
	if(g_eDeviceType == DEVICE_UNAPI)
		return UNAPI_ucConnectToIPAddress(_uiPort, _aucIPAddress);
	else
		return GR8NET_ucConnectToIPAddress(_uiPort, _aucIPAddress);
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
unsigned char DRIVER_ucConnectToHostName(unsigned int _uiPort, char *_szHostName)
{
	if(g_eDeviceType == DEVICE_UNAPI)
		return UNAPI_ucConnectToHostName(_uiPort, _szHostName);
	else
		return GR8NET_ucConnectToHostName(_uiPort, _szHostName);
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void DRIVER_vCloseTCPConnection(char _cConnection)
{
	if(g_eDeviceType == DEVICE_UNAPI)
		UNAPI_vCloseTCPConnection(_cConnection);
	else
		GR8NET_vCloseTCPConnection(_cConnection);
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void DRIVER_vReceiveWithCheck(int _iMaxSize, char *_pcBuffer, char _cConnection, unsigned int *_piBytesRead)
{
	*_piBytesRead = 0;
	if(g_eDeviceType == DEVICE_UNAPI)
		UNAPI_vReceiveWithCheck(_iMaxSize, _pcBuffer, _cConnection, _piBytesRead);
	else
		GR8NET_vReceiveWithCheck(_iMaxSize, _pcBuffer, _cConnection, _piBytesRead);
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
unsigned int DRIVER_uiSendWithCheck(char _cConnection, char *_pcBuffer, unsigned int _uiSize)
{
	if(g_eDeviceType == DEVICE_UNAPI)
		return UNAPI_uiSendWithCheck(_cConnection, _pcBuffer, _uiSize);
	else
		return GR8NET_uiSendWithCheck(_cConnection, _pcBuffer, _uiSize);
}

void to_lowercase(char *str)
{
    while (*str) {
        *str = tolower(*str);
        str++;
    }
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vPutWaitAnimation(unsigned int _uiSize)
{
	/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
	static bool sbPreviousZero = false;
	/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

	if(_uiSize)
	{
		if(sbPreviousZero)
		{
			putch(' ');
			putch(29);
		}

		sbPreviousZero = false;
	}
	else
	{
		/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
		static unsigned char	scState = 0;
		/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

		putch(((scState++) & 1) ? '*' : '+');
		putch(29);
		sbPreviousZero = true;
	}
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vSendStringWithCheck(char *_szString)
{
	if(_szString)
	{
		/*~~~~~~~~~~~~~~~~~~~~~~~~~~*/
		char			*szUnixString;
		char			*pcC;
		unsigned int	uiBytesToSend;
		/*~~~~~~~~~~~~~~~~~~~~~~~~~~*/

		szUnixString = strdup(_szString);
		for(pcC = szUnixString; *pcC; pcC++)
		{
			if(*pcC == '\\') *pcC = '/';
		}

		if(g_bVerbose) cputs(szUnixString);

		uiBytesToSend = strlen(szUnixString);
		pcC = szUnixString;

		while(uiBytesToSend)
		{
			/*~~~~~~~~~~~~~~~~~~~~~~~~*/
			unsigned int	uiBytesSent;
			/*~~~~~~~~~~~~~~~~~~~~~~~~*/

			uiBytesSent = DRIVER_uiSendWithCheck(g_ucConnection0, pcC, uiBytesToSend);
			vPutWaitAnimation(uiBytesSent);
			uiBytesToSend -= uiBytesSent;
			pcC += uiBytesSent;
		}

		free(szUnixString);
	}
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
bool bGetStatusLine(bool _bOnlyOneLine)
{
	/*~~~~~~~~~~~~*/
	bool	bResult;
	/*~~~~~~~~~~~~*/

	bResult = false;

	do
	{
		/*~~~~~~~~~~~~~~*/
		char	cC;
		char	*pcStatus;
		/*~~~~~~~~~~~~~~*/

		pcStatus = g_acStatusLine;
		g_acStatusLine[3] = 0;

		do
		{
			while(!g_iStatusBytesAvailable)
			{
				DRIVER_vReceiveWithCheck
				(
					STATUS_BUFFER_SIZE,
					g_acStatusBuffer,
					g_ucConnection0,
					&g_iStatusBytesAvailable
				);
				vPutWaitAnimation(g_iStatusBytesAvailable);
				g_pcStatus = g_acStatusBuffer;
			}

			cC = *g_pcStatus++;
			*pcStatus++ = cC;
			g_iStatusBytesAvailable--;
		} while(cC != '\n');

		*pcStatus = 0;

		if((*g_acStatusLine >= '1') && (*g_acStatusLine <= '3'))
		{
			if(g_bVerbose) cputs(g_acStatusLine);
			bResult = true;
		}
		else if((*g_acStatusLine >= '4') && (*g_acStatusLine <= '6'))
		{
			if(g_bDisplayErrors) cputs(g_acStatusLine);
		}
		else
		{
			if(!_bOnlyOneLine) cputs(g_acStatusLine);
		}
	} while(!_bOnlyOneLine && ((*g_acStatusLine < '0') || (*g_acStatusLine > '9') || (g_acStatusLine[3] == '-')));

	return bResult;
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
bool bSetBinaryMode()
{
	/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
	static bool s_bBinaryModeSet = false;
	/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

	if(!s_bBinaryModeSet)
	{
		vSendStringWithCheck("type i\r\n");
		if(bGetStatusLine(false))
		{
			s_bBinaryModeSet = true;
		}
		else
		{
			vWarning("Unable to set binary mode", NULL);
		}
	}

	return s_bBinaryModeSet;
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
char cGetCommandNumber(char *_szCommand)
{
	/*~~~~~~~*/
	char	cI;
	/*~~~~~~~*/

	for(cI = 0; cI < (sizeof(g_acSupportedCommands) / sizeof(*g_acSupportedCommands)); cI++)
	{
		if(!stricmp(_szCommand, g_acSupportedCommands[cI])) return cI;
	}

	return -1;
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vGetPortAndIPAddress(char *_szStatusLine, unsigned int *_puiPort, unsigned char *_aucIPAddress)
{
	/*~~~~~~~~~~~~~*/
	char	*szField;
	char	cIndex;
	/*~~~~~~~~~~~~~*/

	memset(_aucIPAddress, 0, 4);
	*_puiPort = 0;

	for(cIndex = 0, szField = strtok(_szStatusLine, ",("); szField; cIndex++, szField = strtok(NULL, ",("))
	{
		/*~~~~~~~~~~~~~~~~~~~~*/
		unsigned int	uiValue;
		/*~~~~~~~~~~~~~~~~~~~~*/

		uiValue = atoi(szField);

		switch(cIndex)
		{
		case 1:
		case 2:
		case 3:
		case 4:
			_aucIPAddress[cIndex - 1] = uiValue;
			break;

		case 5:
			*_puiPort = uiValue << 8;
			break;

		case 6:
			*_puiPort += uiValue;
			break;

		default:
			break;
		}
	}
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vCloseConnection1IfNeeded()
{
	if(g_ucConnection1)
	{
		DRIVER_vCloseTCPConnection(g_ucConnection1);
		g_ucConnection1 = 0;
	}
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vCloseConnection0IfNeeded()
{
	if(g_ucConnection0)
	{
		DRIVER_vCloseTCPConnection(g_ucConnection0);
		g_ucConnection0 = 0;
	}
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
bool bSetPassiveMode()
{
	/*~~~~~~~~~~~~~~~~~~~*/
	unsigned int	uiPort;
	/*~~~~~~~~~~~~~~~~~~~*/

	vSendStringWithCheck("pasv\r\n");
	if(bGetStatusLine(false))
	{
		/*~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
		unsigned char	acIPAddress[4];
		/*~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

		vGetPortAndIPAddress(g_acStatusLine, &uiPort, acIPAddress);

		do
		{
			g_ucConnection1 = DRIVER_ucConnectToIPAddress(uiPort, acIPAddress);
			vPutWaitAnimation(g_ucConnection1);
		} while(!g_ucConnection1);

		return true;
	}
	else
	{
		vWarning("Unable to set passive mode", NULL);
		return false;
	}
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vRemoteDir(char *_szDirectory)
{
	if(bSetPassiveMode())
	{
		vSendStringWithCheck("list ");
		vSendStringWithCheck(_szDirectory);
		vSendStringWithCheck("\r\n");

		if(bGetStatusLine(false))
		{
			/*~~~~~~~~~~~~~~~~~~~~~~~~*/
			bool			bDataClosed;
			unsigned int	uiSize;
			/*~~~~~~~~~~~~~~~~~~~~~~~~*/

			bDataClosed = false;
			do
			{
				do
				{
					DRIVER_vReceiveWithCheck(256, g_acBuffer, g_ucConnection1, &uiSize);
					g_acBuffer[uiSize] = 0;
					cputs(g_acBuffer);
				} while(uiSize);

				bGetStatusLine(false);
				if(g_acStatusLine[0] == '2') bDataClosed = true;
			} while(!bDataClosed);

			vCloseConnection1IfNeeded();
		}
		else
		{
			vWarning("Unable to list directory", NULL);
		}
	}
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vLocalDir(char *_szDirectory)
{
	/*~~~~~~~~~~~~~~~~~~~*/
	tdFileInfoBlock *poFIB;
	/*~~~~~~~~~~~~~~~~~~~*/

	poFIB = malloc(sizeof(*poFIB));

	if(!findfirst("", poFIB))
	{
		/*~~~~~~~~~~~~~~~*/
		char	acSize[11];
		/*~~~~~~~~~~~~~~~*/

		do
		{
			/*~~~~~~~~~~~~~~~~~~~*/
			unsigned long	ulTime;
			char			cLen;
			/*~~~~~~~~~~~~~~~~~~~*/

			cputs(poFIB->m_acFileName);
			for(cLen = strlen(poFIB->m_acFileName); cLen < 12; cLen++) putch(' ');
			putch(' ');
			if(poFIB->m_cAttributes & 16)
			{
				cputs("     <dir>");
			}
			else
			{
				ultoa(poFIB->m_ulFileSize, acSize, 10);
				for(cLen = strlen(acSize); cLen < 10; cLen++) putch(' ');
				cputs(acSize);
			}

			putch(' ');

			ulTime = DOS_ulGetFileTime(poFIB->m_acFileName);

			vPrintDec((ulTime >> 16) & 31, 2);
			putch('-');
			vPrintDec((ulTime >> 21) & 15, 2);
			putch('-');
			vPrintDec((((ulTime >> 25) & 127) + 1980) % 100, 2);
			putch(' ');
			vPrintDec((ulTime >> 11) & 31, 2);
			putch(':');
			vPrintDec((ulTime >> 5) & 63, 2);
			puts("");
		} while(!findnext(poFIB));
	}
	else
	{
		vWarning("Unable to list directory", NULL);
	}

	free(poFIB);
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vMakeRemoteDirectory(char *_szName)
{
	g_bDisplayErrors = false;
	vSendStringWithCheck("mkd ");

    if (g_bForceLowerCase)
        to_lowercase(_szName);

	vSendStringWithCheck(_szName);
	vSendStringWithCheck("\r\n");
	bGetStatusLine(false);
	g_bDisplayErrors = true;
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vMakeLocalDirectory(char *_szName)
{
	/*~~~~~~~~~~~~*/
	char	cResult;
	/*~~~~~~~~~~~~*/

	DOS_disableErrorCheck();
	cResult = mkdir(_szName);
	DOS_enableErrorCheck();

	switch(cResult)
	{
	case 0:
	case -52:
		break;

	default:
		vWarning("Unable to create directory ", _szName);
		break;
	}
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
bool bSetLocalDirectory(char *_szDirectory)
{
	/*~~~~~~~~~~~~~~~~~~~*/
	bool	bResult = true;
	/*~~~~~~~~~~~~~~~~~~~*/

	if(_szDirectory && *_szDirectory)
	{
		if(_szDirectory[1] == ':') DOS_setCurrentDrive(toupper(*_szDirectory) - 'A');

		if(DOS_changeCurrentDirectory(_szDirectory))
		{
			bResult = false;
			vWarning("Unable to change local directory to ", _szDirectory);
		}
	}

	return bResult;
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
bool bCreateAndSetLocalDirectory(char *_szDirectory)
{
	vMakeLocalDirectory(_szDirectory);
	return bSetLocalDirectory(_szDirectory);
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
bool bSetRemoteDirectory(char *_szDirectory)
{
	/*~~~~~~~~~~~~*/
	bool	bResult;
	/*~~~~~~~~~~~~*/

	bResult = true;

	if(_szDirectory)
	{
		vSendStringWithCheck("cwd ");
		vSendStringWithCheck(_szDirectory);
		vSendStringWithCheck("\r\n");
		bResult = bGetStatusLine(false);
		if(!bResult) vWarning("Unable to change remote directory to ", _szDirectory);
	}

	return bResult;
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
bool bCreateAndSetRemoteDirectory(char *_szDirectory)
{
	vMakeRemoteDirectory(_szDirectory);
	return bSetRemoteDirectory(_szDirectory);
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
tdEntry *poGetEntryFromEntries(tdEntry *_poEntries, char *_szName)
{
	/*~~~~~~~~~~~~~*/
	tdEntry *poEntry;
	/*~~~~~~~~~~~~~*/

	for(poEntry = _poEntries; poEntry; poEntry = poEntry->m_poNext)
	{
		if(!stricmp(poEntry->m_acName, _szName))
		{
			return poEntry;
		}
	}

	return NULL;
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
tdEntry *poBasicGetRemoteEntries(char *_szDirectory, char *_szFile)
{
	/*~~~~~~~~~~~~~~~~~~~~~*/
	struct ftpparse oParse;
	tdEntry			*poEntry;
	tdEntry			*poTail;
	tdEntry			*poHead;
	/*~~~~~~~~~~~~~~~~~~~~~*/

	poEntry = NULL;
	poHead = NULL;
	poTail = NULL;

	if(bSetPassiveMode())
	{
		vSendStringWithCheck("list ");
		vSendStringWithCheck(_szDirectory);
		vSendStringWithCheck("\r\n");

		if(bGetStatusLine(false))
		{
			/*~~~~~~~~~~~~~~~~~~~~~~~~*/
			char			*pcDataLine;
			bool			bDataClosed;
			unsigned int	uiSize;
			/*~~~~~~~~~~~~~~~~~~~~~~~~*/

			cputs("Scanning directory... ");

			bDataClosed = false;
			pcDataLine = g_acDataLine;
			do
			{
				do
				{
					/*~~~~~~~~~~~~~~~~~~~~*/
					unsigned int	uiIndex;
					/*~~~~~~~~~~~~~~~~~~~~*/

					DRIVER_vReceiveWithCheck(256, g_acBuffer, g_ucConnection1, &uiSize);
					vPutWaitAnimation(uiSize);

					for(uiIndex = 0; uiIndex < uiSize; uiIndex++)
					{
						*pcDataLine = g_acBuffer[uiIndex];
						if(*pcDataLine == '\n')
						{
							*pcDataLine = 0;
							pcDataLine = g_acDataLine;

							ftpparse(&oParse, g_acDataLine, strlen(g_acDataLine));

							if(oParse.name[0] && strcmp(oParse.name, ".") && strcmp(oParse.name, ".."))
							{
								if((!_szFile) || ((!stricmp(oParse.name, _szFile) && (!poEntry))))
								{
									/*~~~~~~~~~~~~~~~~~~*/
									struct tm	*filetime;
									/*~~~~~~~~~~~~~~~~~~*/

									filetime = gmtime(&oParse.mtime);

									poEntry = malloc(strlen(oParse.name) + 1 + sizeof(*poEntry));
									poEntry->m_poNext = NULL;
									if(!poHead) poHead = poEntry;

									if(!poTail)
									{
										poTail = poEntry;
									}
									else
									{
										poTail->m_poNext = poEntry;
										poTail = poEntry;
									}

									strcpy(poEntry->m_acName, oParse.name);
									poEntry->m_bIsDirectory = oParse.flagtrycwd;
									poEntry->m_ulSize = oParse.size;

									filetime->tm_mon++;

									filetime->tm_min += g_cMinutesAdjust;
									if(filetime->tm_min > 59)
									{
										filetime->tm_hour++;
										filetime->tm_min -= 60;
									}

									if(filetime->tm_min < 0)
									{
										filetime->tm_hour--;
										filetime->tm_min += 60;
									}

									filetime->tm_hour += g_cHoursAdjust;
									while(filetime->tm_hour > 23)
									{
										filetime->tm_hour -= 24;
										if(++filetime->tm_mday > cDaysInAMonth(filetime->tm_year, filetime->tm_mon))
										{
											filetime->tm_mday = 1;
											if(++filetime->tm_mon > 12)
											{
												filetime->tm_mon = 1;
												filetime->tm_year++;
											}
										}
									}

									while(filetime->tm_hour < 0)
									{
										filetime->tm_hour += 24;
										if(--filetime->tm_mday < 1)
										{
											if(--filetime->tm_mon < 1)
											{
												filetime->tm_mon = 12;
												filetime->tm_year--;
											}

											filetime->tm_mday = cDaysInAMonth(filetime->tm_year, filetime->tm_mon);
										}
									}

									poEntry->m_ulFileTime = 0;
									poEntry->m_ulFileTime |= (unsigned long) (filetime->tm_year - 80) << 25;
									poEntry->m_ulFileTime |= (unsigned long) (filetime->tm_mon) << 21;
									poEntry->m_ulFileTime |= (unsigned long) (filetime->tm_mday) << 16;
									poEntry->m_ulFileTime |= (unsigned long) (filetime->tm_hour) << 11;
									poEntry->m_ulFileTime |= (unsigned long) (filetime->tm_min) << 5;
									poEntry->m_ulFileTime |= (unsigned long) (filetime->tm_sec) << 0;
								}
							}
						}
						else if(*pcDataLine != '\r')
						{
							pcDataLine++;
						}
					}
				} while(uiSize);

				bGetStatusLine(false);
				if(g_acStatusLine[0] == '2') bDataClosed = true;
			} while(!bDataClosed);

			puts("");

			vCloseConnection1IfNeeded();
		}
		else
		{
			vWarning("Unable to get file list for ", _szDirectory);
		}
	}

	return poHead;
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
tdEntry *poGetLocalEntries(char *_szDirectory)
{
	/*~~~~~~~~~~~~~~~~~~~~~*/
	tdEntry			*poEntry;
	tdEntry			*poTail;
	tdEntry			*poHead;
	tdFileInfoBlock *poFIB;
	/*~~~~~~~~~~~~~~~~~~~~~*/

	poEntry = NULL;
	poHead = NULL;
	poTail = NULL;

	poFIB = malloc(sizeof(*poFIB));

	if(!findfirst(_szDirectory, poFIB))
	{
		do
		{
			if(strcmp(poFIB->m_acFileName, ".") && strcmp(poFIB->m_acFileName, ".."))
			{
				poEntry = malloc(strlen(poFIB->m_acFileName) + 1 + sizeof(*poEntry));
				poEntry->m_poNext = NULL;
				if(!poHead) poHead = poEntry;
				if(!poTail)
				{
					poTail = poEntry;
				}
				else
				{
					poTail->m_poNext = poEntry;
					poTail = poEntry;
				}

				strcpy(poEntry->m_acName, poFIB->m_acFileName);
				if(poFIB->m_cAttributes & 16)
				{
					poEntry->m_bIsDirectory = true;
				}
				else
				{
					poEntry->m_bIsDirectory = false;
					poEntry->m_ulSize = poFIB->m_ulFileSize;
					poEntry->m_ulFileTime = DOS_ulGetFileTime(poFIB->m_acFileName);
				}
			}
		} while(!findnext(poFIB));
	}
	else
	{
		vWarning("Unable to get file list for ", _szDirectory);
	}

	free(poFIB);

	return poHead;
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
tdEntry *poGetLocalEntry(char *_szFile)
{
	/*~~~~~~~~~~~~~~*/
	FILE	*poFile;
	tdEntry *poResult;
	/*~~~~~~~~~~~~~~*/

	poResult = NULL;

	if(poFile = fopen(_szFile, "rb"))
	{
		poResult = malloc(strlen(_szFile) + 1 + sizeof(*poResult));

		poResult->m_bIsDirectory = false;

		fseek(poFile, 0, SEEK_END);
		poResult->m_ulSize = ftell(poFile);
		fclose(poFile);

		poResult->m_ulFileTime = DOS_ulGetFileTime(_szFile);
		strcpy(poResult->m_acName, _szFile);
	}

	return poResult;
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
tdEntry *poGetRemoteEntries(char *_szDirectory)
{
	return poBasicGetRemoteEntries(_szDirectory, NULL);
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
tdEntry *poGetRemoteEntry(char *_szFile)
{
	/*~~~~~~~~~~~~~~~~~*/
	char	*pcLastSlash;
	char	*pcC;
	tdEntry *poResult;
	/*~~~~~~~~~~~~~~~~~*/

	pcLastSlash = NULL;

	for(pcC = _szFile; *pcC; pcC++)
	{
		if((*pcC == '/') || (*pcC == '\\')) pcLastSlash = pcC;
	}

	if(pcLastSlash)
	{
		/*~~~~~~~~~~~~*/
		char	cBackup;
		/*~~~~~~~~~~~~*/

		cBackup = *pcLastSlash;
		*pcLastSlash = 0;
		poResult = poBasicGetRemoteEntries(_szFile, pcLastSlash + 1);
		*pcLastSlash = cBackup;
	}
	else
	{
		poResult = poBasicGetRemoteEntries("", _szFile);
	}

	return poResult;
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vBasicPutFile(char *_szLocalName, char *_szRemoteName, unsigned long _ulSizeTransfer)
{
	if(g_poFile = poOpenFileWithCheck(_szLocalName, "rb"))
	{
		if(bSetPassiveMode())
		{
            if (g_bForceLowerCase)
            {
                to_lowercase(_szRemoteName);
            }

			vSendStringWithCheck("stor ");
			vSendStringWithCheck(_szRemoteName);
			vSendStringWithCheck("\r\n");

			if(bGetStatusLine(false))
			{
				/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
				unsigned int	uiSize;
				unsigned long	ulTotalSize;
				unsigned long	ulPercentageSize;
				long			lPercentageBytes;
				unsigned char	ucPercentage;
				/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

				cputs("Putting ");
				cputs(_szLocalName);
				putch(' ');

				fseek(g_poFile, 0, SEEK_END);
				ulTotalSize = ftell(g_poFile);
				fseek(g_poFile, 0, SEEK_SET);

				ulPercentageSize = (ulTotalSize + 50) / 100;
				if(!ulPercentageSize) ulPercentageSize = 1;
				lPercentageBytes = 0;
				ucPercentage = 0;

				DOS_disableErrorCheck();

				while(uiSize = fread(g_acTransferBuffer, 1, TRANSFER_BLOCK_SIZE, g_poFile))
				{
					/*~~~~~~~~~~~~~~~~~~~~~~~~~~*/
					char			*pcC;
					unsigned int	uiBytesToSend;
					/*~~~~~~~~~~~~~~~~~~~~~~~~~~*/

					uiBytesToSend = uiSize;
					pcC = g_acTransferBuffer;

					while(uiBytesToSend)
					{
						/*~~~~~~~~~~~~~~~~~~~~~~~~*/
						unsigned int	uiBytesSent;
						/*~~~~~~~~~~~~~~~~~~~~~~~~*/

						uiBytesSent = DRIVER_uiSendWithCheck(g_ucConnection1, pcC, uiBytesToSend);
						vPutWaitAnimation(uiBytesSent);

						uiBytesToSend -= uiBytesSent;
						pcC += uiBytesSent;
					}

					if(uiSize > lPercentageBytes)
					{
						while(uiSize > lPercentageBytes)
						{
							lPercentageBytes += ulPercentageSize;
							ucPercentage++;
						}

						vDisplayPercentage(ucPercentage);
					}

					lPercentageBytes -= uiSize;
				}

				DOS_enableErrorCheck();

				puts(" OK  ");

				vCloseConnection1IfNeeded();
				bGetStatusLine(false);
			}
		}

		fclose(g_poFile);
		g_poFile = NULL;
	}
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vBasicGetFile(char *_szRemoteName, char *_szLocalName, unsigned long _ulSizeToTransfer)
{
	if(g_poFile = poOpenFileWithCheck(_szLocalName, "wb"))
	{
		/*~~~~~~~~~~~*/
		char	cRetry;
		/*~~~~~~~~~~~*/

		cputs("Getting ");
		cputs(_szRemoteName);
		putch(' ');

		for(cRetry = 0; cRetry < 16; cRetry++)
		{
			if(_ulSizeToTransfer)
			{
				fseek(g_poFile, _ulSizeToTransfer - 1l, SEEK_SET);
				fwrite((void *) 0x8000, 1, 1, g_poFile);
				fensure(g_poFile);
				fseek(g_poFile, 0, SEEK_SET);
			}
			if(cRetry) vWarning("File not ready, retrying...", NULL);

			if(bSetPassiveMode())
			{
				vSendStringWithCheck("retr ");
				vSendStringWithCheck(_szRemoteName);
				vSendStringWithCheck("\r\n");

				if(bGetStatusLine(false))
				{
					/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
					unsigned int	uiSize;
					unsigned long	ulBytesRead;
					unsigned long	ulPercentageSize;
					long			lPercentageBytes;
					unsigned char	ucPercentage;
					bool			bDataClosed;
					/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

					ulBytesRead = 0;
					lPercentageBytes = 0;
					ucPercentage = 0;
					bDataClosed = false;

					ulPercentageSize = (_ulSizeToTransfer + 50) / 100;
					if(!ulPercentageSize) ulPercentageSize = 1;

					while(ulBytesRead < _ulSizeToTransfer)
					{
						DRIVER_vReceiveWithCheck(TRANSFER_BLOCK_SIZE, g_acTransferBuffer, g_ucConnection1, &uiSize);
						if(uiSize)
						{
							ulBytesRead += uiSize;
							fwrite(g_acTransferBuffer, 1, uiSize, g_poFile);
							if(uiSize > lPercentageBytes)
							{
								while(uiSize > lPercentageBytes)
								{
									lPercentageBytes += ulPercentageSize;
									ucPercentage++;
								}

								vDisplayPercentage(ucPercentage);
							}

							lPercentageBytes -= uiSize;
						}

						vPutWaitAnimation(uiSize);
					}

					puts(" OK  ");
					vCloseConnection1IfNeeded();
					bGetStatusLine(false);

					break;
				}
			}
		}

		fclose(g_poFile);
		g_poFile = NULL;
	}
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vPutFile(tdEntry *_poLocalEntry, char *_szRemoteName)
{
	if(!_szRemoteName) _szRemoteName = _poLocalEntry->m_acName;

	vBasicPutFile(_poLocalEntry->m_acName, _szRemoteName, _poLocalEntry->m_ulSize);
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vGetFile(tdEntry *_poRemoteEntry, char *_szLocalName)
{
	if(!_szLocalName) _szLocalName = _poRemoteEntry->m_acName;

	_szLocalName = szFixFileName(_szLocalName);

	vBasicGetFile(_poRemoteEntry->m_acName, _szLocalName, _poRemoteEntry->m_ulSize);
	DOS_cSetFileTime(_szLocalName, _poRemoteEntry->m_ulFileTime);

	free(_szLocalName);
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
bool bIsNewer(tdEntry *_poA, tdEntry *_poB)
{
	return _poA->m_ulFileTime > _poB->m_ulFileTime;
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vPutFileUpdateOnly(char *_szLocalName, char *_szRemoteName)
{
	/*~~~~~~~~~~~~~~~~~~*/
	tdEntry *poLocalEntry;
	/*~~~~~~~~~~~~~~~~~~*/

	if(poLocalEntry = poGetLocalEntry(_szLocalName))
	{
		/*~~~~~~~~~~~~~~~~~~~*/
		tdEntry *poRemoteEntry;
		/*~~~~~~~~~~~~~~~~~~~*/

		if(!_szRemoteName) _szRemoteName = _szLocalName;

		poRemoteEntry = poGetRemoteEntry(_szRemoteName);

		if(!poRemoteEntry || bIsNewer(poLocalEntry, poRemoteEntry))
		{
			vPutFile(poLocalEntry, _szRemoteName);
		}
		else
		{
			cputs("Skipping ");
			puts(_szLocalName);
		}

		if(poRemoteEntry) free(poRemoteEntry);
		free(poLocalEntry);
	}
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vGetFileUpdateOnly(char *_szRemoteName, char *_szLocalName)
{
	/*~~~~~~~~~~~~~~~~~~~*/
	tdEntry *poRemoteEntry;
	/*~~~~~~~~~~~~~~~~~~~*/

	poRemoteEntry = poGetRemoteEntry(_szRemoteName);

	if(poRemoteEntry)
	{
		/*~~~~~~~~~~~~~~~~~~*/
		tdEntry *poLocalEntry;
		/*~~~~~~~~~~~~~~~~~~*/

		if(!_szLocalName) _szLocalName = _szRemoteName;

		poLocalEntry = poGetLocalEntry(_szLocalName);
		_szLocalName = szFixFileName(_szLocalName);

		if(!poLocalEntry || bIsNewer(poRemoteEntry, poLocalEntry))
		{
			vGetFile(poRemoteEntry, _szLocalName);
		}
		else
		{
			cputs("Skipping ");
			puts(_szRemoteName);
		}

		if(poLocalEntry) free(poLocalEntry);
		free(poRemoteEntry);
		free(_szLocalName);
	}
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vPutDirectory()
{
	/*~~~~~~~~~~~~~~~~~~~~*/
	tdEntry *poLocalEntries;
	tdEntry *poLocalEntry;
	/*~~~~~~~~~~~~~~~~~~~~*/

	poLocalEntries = poGetLocalEntries("");

	for(poLocalEntry = poLocalEntries; poLocalEntry; free(poLocalEntry), poLocalEntry = poLocalEntry->m_poNext)
	{
		if(poLocalEntry->m_bIsDirectory)
		{
			cputs("cd ");
			puts(poLocalEntry->m_acName);

			if(bCreateAndSetRemoteDirectory(poLocalEntry->m_acName))
			{
				if(bSetLocalDirectory(poLocalEntry->m_acName))
				{
					vPutDirectory();
					bSetLocalDirectory("..");
				}

				bSetRemoteDirectory("..");
			}

			puts("cd ..");
		}
		else
		{
			vPutFile(poLocalEntry, NULL);
		}
	}
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vGetDirectory()
{
	/*~~~~~~~~~~~~~~~~~~~~~*/
	tdEntry *poRemoteEntries;
	tdEntry *poRemoteEntry;
	/*~~~~~~~~~~~~~~~~~~~~~*/

	poRemoteEntries = poGetRemoteEntries(".");

	for(poRemoteEntry = poRemoteEntries; poRemoteEntry; free(poRemoteEntry), poRemoteEntry = poRemoteEntry->m_poNext)
	{
		if(poRemoteEntry->m_bIsDirectory)
		{
			cputs("cd ");
			puts(poRemoteEntry->m_acName);

			if(bSetRemoteDirectory(poRemoteEntry->m_acName))
			{
				/*~~~~~~~~~~~~~~~~~~~~~~~~~~*/
				char	*szLocalDirectoryName;
				/*~~~~~~~~~~~~~~~~~~~~~~~~~~*/

				szLocalDirectoryName = szFixFileName(poRemoteEntry->m_acName);
				if(bCreateAndSetLocalDirectory(szLocalDirectoryName))
				{
					vGetDirectory();
					bSetLocalDirectory("..");
				}

				free(szLocalDirectoryName);

				bSetRemoteDirectory("..");
			}

			puts("cd ..");
		}
		else
		{
			vGetFile(poRemoteEntry, NULL);
		}
	}
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vPutDirectoryUpdateOnly()
{
	/*~~~~~~~~~~~~~~~~~~~~*/
	tdEntry *poLocalEntries;
	tdEntry *poLocalEntry;
	/*~~~~~~~~~~~~~~~~~~~~*/

	poLocalEntries = poGetLocalEntries("");

	if(poLocalEntries)
	{
		/*~~~~~~~~~~~~~~~~~~~~~*/
		tdEntry *poRemoteEntries;
		tdEntry *poRemoteEntry;
		/*~~~~~~~~~~~~~~~~~~~~~*/

		poRemoteEntries = poGetRemoteEntries(".");

		for(poLocalEntry = poLocalEntries; poLocalEntry; free(poLocalEntry), poLocalEntry = poLocalEntry->m_poNext)
		{
			if(poLocalEntry->m_bIsDirectory)
			{
				cputs("cd ");
				puts(poLocalEntry->m_acName);

				if(bCreateAndSetRemoteDirectory(poLocalEntry->m_acName))
				{
					if(bSetLocalDirectory(poLocalEntry->m_acName))
					{
						vPutDirectoryUpdateOnly();
						bSetLocalDirectory("..");
					}

					bSetRemoteDirectory("..");
				}

				puts("cd ..");
			}
			else
			{
				poRemoteEntry = poGetEntryFromEntries(poRemoteEntries, poLocalEntry->m_acName);
				if(poRemoteEntry)
				{
					if(bIsNewer(poLocalEntry, poRemoteEntry))
					{
						vPutFile(poLocalEntry, NULL);
					}
					else
					{
						cputs("Skipping ");
						puts(poLocalEntry->m_acName);
					}
				}
				else
				{
					vPutFile(poLocalEntry, NULL);
				}
			}
		}

		for
		(
			poRemoteEntry = poRemoteEntries;
			poRemoteEntry;
			free(poRemoteEntry), poRemoteEntry = poRemoteEntry->m_poNext
		);
	}
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vGetDirectoryUpdateOnly()
{
	/*~~~~~~~~~~~~~~~~~~~~~*/
	tdEntry *poRemoteEntries;
	tdEntry *poRemoteEntry;
	/*~~~~~~~~~~~~~~~~~~~~~*/

	poRemoteEntries = poGetRemoteEntries(".");

	if(poRemoteEntries)
	{
		/*~~~~~~~~~~~~~~~~~~~~*/
		tdEntry *poLocalEntries;
		tdEntry *poLocalEntry;
		/*~~~~~~~~~~~~~~~~~~~~*/

		poLocalEntries = poGetLocalEntries("");

		for
		(
			poRemoteEntry = poRemoteEntries;
			poRemoteEntry;
			free(poRemoteEntry), poRemoteEntry = poRemoteEntry->m_poNext
		)
		{
			if(poRemoteEntry->m_bIsDirectory)
			{
				cputs("cd ");
				puts(poRemoteEntry->m_acName);

				if(bSetRemoteDirectory(poRemoteEntry->m_acName))
				{
					/*~~~~~~~~~~~~~~~~~~~~~~~~~~*/
					char	*szLocalDirectoryName;
					/*~~~~~~~~~~~~~~~~~~~~~~~~~~*/

					szLocalDirectoryName = szFixFileName(poRemoteEntry->m_acName);
					if(bCreateAndSetLocalDirectory(szLocalDirectoryName))
					{
						vGetDirectoryUpdateOnly();
						bSetLocalDirectory("..");
					}

					free(szLocalDirectoryName);

					bSetRemoteDirectory("..");
				}

				puts("cd ..");
			}
			else
			{
				poLocalEntry = poGetEntryFromEntries(poLocalEntries, poRemoteEntry->m_acName);
				if(poLocalEntry)
				{
					if(bIsNewer(poRemoteEntry, poLocalEntry))
					{
						vGetFile(poRemoteEntry, poLocalEntry->m_acName);
					}
					else
					{
						cputs("Skipping ");
						puts(poRemoteEntry->m_acName);
					}
				}
				else
				{
					vGetFile(poRemoteEntry, NULL);
				}
			}
		}

		for(poLocalEntry = poLocalEntries; poLocalEntry; free(poLocalEntry), poLocalEntry = poLocalEntry->m_poNext);
	}
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
char *szGetFTPCurrentDirectory()
{
	vSendStringWithCheck("pwd\r\n");
	if(bGetStatusLine(false))
	{
		strtok(g_acStatusLine, "\"");
		return strtok(NULL, "\"");
	}
	else
		return NULL;
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vSendCommand(char *_szCommandLine)
{
	/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
	char	*_szCommand;
	char	*_szArgument;
	char	*_szArgument2;
	char	cCommand;
	char	cCurrentLocalDrive;
	char	*pcCurrentLocalDirectory;
	char	*pcCurrentFTPDirectory;
	bool	bRecursiveCommand;
	/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

	_szCommand = strtok(_szCommandLine, " \r");
	_szArgument = strtok(NULL, " \r");
	_szArgument2 = strtok(NULL, " \r");

	pcCurrentLocalDirectory = NULL;
	pcCurrentFTPDirectory = NULL;
	cCurrentLocalDrive = -1;

	cCommand = cGetCommandNumber(_szCommand);

	bRecursiveCommand = (cCommand == COMMAND_GETR) || (cCommand == COMMAND_GETRU) || (cCommand == COMMAND_PUTR)
||	(cCommand == COMMAND_PUTRU);

	if(bRecursiveCommand)
	{
		cCurrentLocalDrive = DOS_getCurrentDrive();

		pcCurrentLocalDirectory = malloc(MAX_PATH + 1);
		*pcCurrentLocalDirectory = '\\';
		DOS_getCurrentDirectory(pcCurrentLocalDirectory + 1, 0);

		pcCurrentFTPDirectory = szGetFTPCurrentDirectory();
		if(pcCurrentFTPDirectory) pcCurrentFTPDirectory = strdup(pcCurrentFTPDirectory);
	}

	switch(cCommand)
	{
	case COMMAND_LCD:
		if(_szArgument)
		{
			bSetLocalDirectory(_szArgument);
		}
		else
		{
			putch('A' + DOS_getCurrentDrive());
			cputs(":\\");
			pcCurrentLocalDirectory = malloc(MAX_PATH);
			DOS_getCurrentDirectory(pcCurrentLocalDirectory, 0);
			puts(pcCurrentLocalDirectory);
			free(pcCurrentLocalDirectory);
		}
		break;

	case COMMAND_LDIR:
		vLocalDir(_szArgument ? _szArgument : ".");
		break;

	case COMMAND_CD:
		if(_szArgument)
			bSetRemoteDirectory(_szArgument);
		else
		{
			/*~~~~~~~~~~~~~~~~~~~~~~~~*/
			char	*szCurrentDirectory;
			/*~~~~~~~~~~~~~~~~~~~~~~~~*/

			szCurrentDirectory = szGetFTPCurrentDirectory();
			if(szCurrentDirectory) puts(szCurrentDirectory);
		}
		break;

	case COMMAND_DIR:
		vRemoteDir(_szArgument);
		break;

	case COMMAND_GET:
		if(bSetBinaryMode())
		{
			/*~~~~~~~~~~~~~~~~~~~*/
			tdEntry *poRemoteEntry;
			/*~~~~~~~~~~~~~~~~~~~*/

			if(poRemoteEntry = poGetRemoteEntry(_szArgument))
			{
				vGetFile(poRemoteEntry, _szArgument2);
				free(poRemoteEntry);
			}
		}
		break;

	case COMMAND_GETU:
		if(bSetBinaryMode()) vGetFileUpdateOnly(_szArgument, _szArgument2);
		break;

	case COMMAND_PUT:
		if(bSetBinaryMode())
		{
			/*~~~~~~~~~~~~~~~~~~*/
			tdEntry *poLocalEntry;
			/*~~~~~~~~~~~~~~~~~~*/

			if(poLocalEntry = poGetLocalEntry(_szArgument))
			{
				vPutFile(poLocalEntry, _szArgument2);
				free(poLocalEntry);
			}
		}
		break;

	case COMMAND_PUTU:
		if(bSetBinaryMode()) vPutFileUpdateOnly(_szArgument, _szArgument2);
		break;

	case COMMAND_HELP:
		vSendStringWithCheck("help\r\n");
		bGetStatusLine(false);
		puts(g_szCommandsList);
		break;

	case COMMAND_QUIT:
		vSendStringWithCheck("quit\r\n");
		bGetStatusLine(false);
		g_bQuit = true;
		break;

	case COMMAND_VERBOSE:
		if(_szArgument)
			g_bVerbose = atoi(_szArgument);
		else
			vWarning("Verbose mode is ", g_bVerbose ? "on" : "off");
		break;

	case COMMAND_LOWER:
		if(_szArgument)
			g_bForceLowerCase = atoi(_szArgument);
		else
			vWarning("Lower case mode is ", g_bForceLowerCase ? "on" : "off");
		break;

	case COMMAND_TA:
		if(_szArgument)
		{
			/*~~~~~~~~~~~~~~~*/
			char	*szHours;
			char	*szMinutes;
			int		iHours;
			int		iMinutes;
			/*~~~~~~~~~~~~~~~*/

			szHours = strtok(_szArgument, ":\r");
			szMinutes = strtok(NULL, " \r");

			iHours = atoi(szHours);
			iMinutes = atoi(szMinutes);

			if((iMinutes > 59) || (iMinutes < -59))
			{
				vWarning("Invalid time", NULL);
			}
			else
			{
				g_cHoursAdjust = iHours;
				g_cMinutesAdjust = iMinutes;
				if(g_cHoursAdjust < 0) g_cMinutesAdjust = -g_cMinutesAdjust;
			}
		}
		else
		{
			cputs("** Time adjustment is ");
			vPrintDec(g_cHoursAdjust, 2);
			cputs(":");
			vPrintDec(g_cHoursAdjust >= 0 ? g_cMinutesAdjust : -g_cMinutesAdjust, 2);
			puts("");
		}
		break;

	case COMMAND_GETR:
		if(bSetBinaryMode() && bSetRemoteDirectory(_szArgument) && bCreateAndSetLocalDirectory(_szArgument2))
			vGetDirectory();
		break;

	case COMMAND_GETRU:
		if(bSetBinaryMode() && bSetRemoteDirectory(_szArgument) && bCreateAndSetLocalDirectory(_szArgument2))
			vGetDirectoryUpdateOnly();
		break;

	case COMMAND_PUTR:
		if(bSetBinaryMode() && bSetLocalDirectory(_szArgument) && bCreateAndSetRemoteDirectory(_szArgument2))
			vPutDirectory();
		break;

	case COMMAND_PUTRU:
		if(bSetBinaryMode() && bSetLocalDirectory(_szArgument) && bCreateAndSetRemoteDirectory(_szArgument2))
			vPutDirectoryUpdateOnly();
		break;

	default:
		if(*_szCommand)
		{
			vSendStringWithCheck(_szCommand);
			if(_szArgument)
			{
				vSendStringWithCheck(" ");
				vSendStringWithCheck(_szArgument);
			}

			vSendStringWithCheck("\r\n");
			bGetStatusLine(false);
		}
		break;
	}

	vCloseConnection1IfNeeded();

	if(bRecursiveCommand)
	{
		if(cCurrentLocalDrive >= 0) DOS_setCurrentDrive(cCurrentLocalDrive);

		if(pcCurrentLocalDirectory)
		{
			DOS_changeCurrentDirectory(pcCurrentLocalDirectory);
			free(pcCurrentLocalDirectory);
		}

		if(pcCurrentFTPDirectory)
		{
			bSetRemoteDirectory(pcCurrentFTPDirectory);
			free(pcCurrentFTPDirectory);
		}
	}
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vOutOfTPA()
{
	vError("Out of TPA memory", NULL);
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vOutOfSegments()
{
	vError("Out of segments", NULL);
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vClean()
{
	DOS_defineDiskAbortRoutine(NULL);
	vCloseConnection0IfNeeded();
	vCloseConnection1IfNeeded();
	if(g_poFile) fclose(g_poFile);
	vRestoreMemoryConfig();
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vError(char *_szMessage1, char *_szMessage2)
{
	vClean();
	cputs("*** ");
	if(_szMessage1) cputs(_szMessage1);
	puts(_szMessage2 ? _szMessage2 : "");
	exit(1);
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vWarning(char *_szMessage1, char *_szMessage2)
{
	cputs("** ");
	if(_szMessage1) cputs(_szMessage1);
	puts(_szMessage2 ? _szMessage2 : "");
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vExit(char _cErrorCode)
{
	_opc(0x5F);
	vClean();
	exit(_cErrorCode);
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
int main(int argc, char *argv[])
{
	/*~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
	int		iIndex;
	bool	bForceUNAPI = false;
	char	*szCommandStart = NULL;
	/*~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

	stacksize = 1024;
	oomcb = vOutOfTPA;
	g_ucCommandCapacity = 255;

	SEG_bInit();
	g_acTransferBuffer = malloc(TRANSFER_BLOCK_SIZE);
	g_acStatusBuffer = malloc(STATUS_BUFFER_SIZE);
	g_acBuffer = malloc(257);
	g_acDataLine = malloc(257);

	vSaveMemoryConfig();
	DOS_defineDiskAbortRoutine(vExit);

	for(iIndex = 1; iIndex < argc; iIndex++)
	{
		/*~~~~~~~~~*/
		char	*szP;
		/*~~~~~~~~~*/

		szP = argv[iIndex];
		if(szP[0] == '/')
		{
			switch(szP[1])
			{
			case 'u':
			case 'U':
				bForceUNAPI = 1;
				break;

			default:
				vError("Invalid option: ", szP);
				break;
			}
		}
		else
		{
			if(!g_szHostName)
			{
				g_szHostName = szP;
			}
			else if(!szCommandStart)
			{
				szCommandStart = szP;
			}
			else
			{
				puts(g_szUsage);
				cputs(g_szCommandsList);
				return 0;
			}
		}
	}

	if(!g_szHostName)
	{
		puts(g_szUsage);
		cputs(g_szCommandsList);
		return 0;
	}

	if(!bForceUNAPI && GR8NET_bInit())
	{
		g_eDeviceType = DEVICE_GR8NET;
	}
	else if(UNAPI_bInit())
	{
		g_eDeviceType = DEVICE_UNAPI;
	}
	else
	{
		vError("No network interface found", NULL);
	}

	g_bVerbose = true;

	do
	{
		g_ucConnection0 = DRIVER_ucConnectToHostName(21, g_szHostName);
		vPutWaitAnimation(g_ucConnection0);
	} while(!g_ucConnection0);

	bGetStatusLine(false);
	g_bVerbose = false;
    g_bForceLowerCase = false;

	if(szCommandStart)
	{
		/*~~~~~~~~~~~~~~~*/
		bool	bEndOfLine;
		/*~~~~~~~~~~~~~~~*/

		bEndOfLine = false;
		while(*szCommandStart && (!g_bQuit) && (!bEndOfLine))
		{
			/*~~~~~~~~~*/
			char	*pcC;
			/*~~~~~~~~~*/

			for(pcC = szCommandStart; (*pcC) && (*pcC != '!'); pcC++)
			{
				if(*pcC == '*') *pcC = ' ';
			}

			if(!*pcC) bEndOfLine = true;

			*pcC = 0;
			strcpy(g_acCommand, szCommandStart);
			vSendCommand(g_acCommand);
			szCommandStart = pcC + 1;
		}
	}

	while(!g_bQuit)
	{
		cputs(">");
		vGetString(&g_ucCommandCapacity);
		g_acCommand[g_ucCommandSize] = 0;
		puts("");
		vSendCommand(g_acCommand);
	}

	vClean();
	return 0;
}
